package com.patzn.lims.common;

import com.google.common.collect.Lists;
import org.apache.commons.lang3.StringUtils;
import org.joda.time.DateTime;

import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.List;
import java.util.Objects;

/**
 * <p>
 * 编号 KEY 生成工具类
 * </p>
 *
 * @author hubin
 * @since 2018-04-11
 */
public class CodeKeyUtils {


    /**
     * <p>
     * 提取内拼音字母
     * </p>
     *
     * @param context      提取内容
     * @param retainLength 保留长度
     * @param toUpperCase  转为大写
     * @return
     */
    public static String getSpell(String context, int retainLength, boolean toUpperCase) {
        if (StringUtils.isBlank(context)) {
            return "";
        }
        String out = Pinyin4jUtils.converterToSpell(context);
        if (retainLength > 0 && retainLength < out.length()) {
            out = out.substring(0, retainLength);
        }
        return toUpperCase ? out.toUpperCase() : out;
    }


    /**
     * <p>
     * 提取内容首拼音字母
     * </p>
     *
     * @param context      提取内容
     * @param retainLength 保留长度
     * @param toUpperCase  转为大写
     * @return
     */
    public static String getFirstSpell(String context, int retainLength, boolean toUpperCase) {
        return getFirstSpell(context, "", retainLength, toUpperCase);
    }


    /**
     * <p>
     * 提取内容首拼音字母<br>
     * getFirstSpell("天津市", 2, false)   // 输出：tj
     * getFirstSpell("天津市", 3, true)    // 输出：TJS
     * getFirstSpell("天津市", -1, true)   // 输出：TJ
     * getFirstSpell(null, "YY", 3, true) // 输出：YY
     * </p>
     *
     * @param context      提取内容
     * @param deftext      默认内容
     * @param retainLength 保留长度
     * @param toUpperCase  转为大写
     * @return
     */
    public static String getFirstSpell(String context, String deftext, int retainLength, boolean toUpperCase) {
        if (StringUtils.isBlank(context)) {
            return null == deftext ? "" : deftext;
        }
        String out = Pinyin4jUtils.converterToFirstSpell(context);
        if (retainLength < 0) {
            // 去除尾部内容
            int length = out.length() + retainLength;
            if (length > 0) {
                out = out.substring(0, length);
            }
        } else if (retainLength > 0 && retainLength < out.length()) {
            // 截图长度
            out = out.substring(0, retainLength);
        }
        return toUpperCase ? out.toUpperCase() : out;
    }


    public static String getSomeCharacter(String context, String deftext, int fromCut) {
        return getSomeCharacter(context, deftext, fromCut, 0);
    }


    public static String getSomeCharacter(String context, String deftext, int fromCut, int toCut) {
        if (StringUtils.isBlank(context) || context.length() < fromCut) {
            return null == deftext ? "" : deftext;
        }
        if (fromCut < 0) {
            // 负数截取尾部内容
            fromCut = context.length() + fromCut;
            if (fromCut < 0) {
                return deftext;
            }
        } else if (toCut > fromCut) {
            return context.substring(fromCut, toCut);
        }
        return context.substring(fromCut);
    }

    /**
     * <p>
     * 获取编号
     * </p>
     *
     * @param sn     序列号
     * @param length 序列号长度
     * @return
     */
    public static String getSn(int sn, int length) {
        StringBuffer out = new StringBuffer();
        // 序号
        if (length > 0) {
            int snl = length - String.valueOf(sn).length();
            if (snl > 0) {
                for (int i = 0; i < snl; i++) {
                    out.append("0");
                }
            }
        }
        out.append(sn);
        return out.toString();
    }

    /**
     * <p>
     * 获取编号,A01-Z99
     * </p>
     *
     * @param sn     序列号
     * @param length 序列号长度
     * @return
     */
    public static String getSnWithLetter(int sn, int length) {
        StringBuffer out = new StringBuffer();
        // 序号
        if (length > 0) {
            int snl = length - String.valueOf(sn).length();
            if (snl > 0) {
                for (int i = 0; i < snl; i++) {
                    out.append("0");
                }
            }
        }
        out.append(sn);
        String result = out.toString();
        String aheadStr = "";
        if(result.length()>2){
            String subStr = result.substring(0, 2);
            switch (subStr){
                case "00":
                    aheadStr = "A";
                    break;
                case "01":
                    aheadStr = "B";
                    break;
                case "02":
                    aheadStr = "C";
                    break;
                case "03":
                    aheadStr = "D";
                    break;
                case "04":
                    aheadStr = "E";
                    break;
                case "05":
                    aheadStr = "F";
                    break;
                case "06":
                    aheadStr = "G";
                    break;
                case "07":
                    aheadStr = "H";
                    break;
                case "08":
                    aheadStr = "I";
                    break;
                case "09":
                    aheadStr = "J";
                    break;
                case "10":
                    aheadStr = "K";
                    break;
                case "11":
                    aheadStr = "L";
                    break;
                case "12":
                    aheadStr = "M";
                    break;
                case "13":
                    aheadStr = "N";
                    break;
                case "14":
                    aheadStr = "O";
                    break;
                case "15":
                    aheadStr = "P";
                    break;
                case "16":
                    aheadStr = "Q";
                    break;
                case "17":
                    aheadStr = "R";
                    break;
                case "18":
                    aheadStr = "S";
                    break;
                case "19":
                    aheadStr = "T";
                    break;
                case "20":
                    aheadStr = "U";
                    break;
                case "21":
                    aheadStr = "V";
                    break;
                case "22":
                    aheadStr = "W";
                    break;
                case "23":
                    aheadStr = "X";
                    break;
                case "24":
                    aheadStr = "Y";
                    break;
                case "25":
                    aheadStr = "Z";
                    break;
                default:
                    aheadStr = "";
                    break;
            }
            String tempResult = result.substring(2, result.length());
            result = aheadStr + tempResult;
        }
        return result;
    }

    public static String getSubStr(String code, int count) {
        if (StringUtils.isNotBlank(code) && code.length() > count) {
            return code.substring(code.length()-count,code.length());
        }
        return "";
    }

    /**
     * <p>
     * 编号日期混合方法，例如 code = 123  日期 01 返回  1 0 2 1 3
     * </p>
     *
     * @param code 编号
     * @return
     */
    public static String getCodeMonth(String code) {
        if (null != code && code.length() > 2) {
            StringBuilder result = new StringBuilder();
            result.append(code, 0, 1);
            String month = DateTime.now().toString("MM");
            result.append(month, 0, 1);
            result.append(code, 1, 2);
            result.append(month.substring(1));
            result.append(code.substring(2));
            return result.toString();
        }
        return code;
    }

    /**
     * 编号日期混合方法，例如 code = 001  日期 201901 返回   0 1 0 9 1 A
     *
     * @param code
     * @return
     */
    public static String getCodeJY(String code) {
        if (null != code && code.length() > 2) {
            DateTime now = DateTime.now();
            String year = now.toString("yy");
            String month = now.toString("MM");
            StringBuilder result = new StringBuilder();
            result.append(code, 0, 1);
            result.append(year, 0, 1);
            result.append(code, 1, 2);
            result.append(year, 1, 2);
            result.append(code.substring(2));
            int monthInt = Integer.valueOf(month);
            char cc = 'A';
            cc = (char) (monthInt - 1 + cc);
            result.append(cc);
            return result.toString();
        }
        return code;
    }

    public static String getRandom(int length) {
        return RandomUtils.getRandomNumber(length);
    }

    public static String getCharByYear() {
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy");
        String year = sdf.format(new Date());
        int yearInt = Integer.valueOf(year);
        char code = 'A';
        code = (char) (yearInt - 2018 + code);
        List<Character> list = Lists.newArrayList('E', 'F', 'Q', 'T', 'Z');
        while (list.contains(code)) {
            code += 1;
        }
        return code + "";
    }

    public static String getCharByMonth() {
        SimpleDateFormat sdf = new SimpleDateFormat("MM");
        String year = sdf.format(new Date());
        int monthInt = Integer.valueOf(year);
        char code = 'A';
        code = (char) (monthInt - 1 + code);
        return code + "";
    }

    public static String getMonthDay(String receiveDate) {
        String result = "";
        if (StringUtils.isNotBlank(receiveDate)) {
            result = receiveDate.substring(5, 7) + receiveDate.substring(8, 10);
        }
        return result;
    }

    public static String getYearMonthDay(String samplingDateLeft) {
        String result = "";
        if (StringUtils.isNotBlank(samplingDateLeft)) {
            result = samplingDateLeft.substring(2,8);
        }
        return result;
    }

    public static String getSampleSnLastFour(String sampleSn) {
        String result = "";
        if (StringUtils.isNotBlank(sampleSn)) {
            result = sampleSn.substring(sampleSn.length() - 4);
        }
        return result;
    }

    public static String getSalesCodeThirdEight(String salesCode) {
        String result = "";
        if (StringUtils.isNotBlank(salesCode) && salesCode.length() >= 8) {
            result = salesCode.substring(2, 8);
        }
        return result;
    }

    public static String getSampleCodeLastSeven(String sampleCode) {
        String result = "";
        if (StringUtils.isNotBlank(sampleCode) && sampleCode.length() >= 7) {
            result = sampleCode.substring(sampleCode.length() - 7, sampleCode.length());
        }
        return result;
    }

    /**
     * @param subCount == 后几位 eg.2019想取019，则参数为1,19-->2
     * @return
     */
    public static String getSubByYear(int subCount) {
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy");
        String year = sdf.format(new Date());
        return year.substring(subCount);
    }

    /**
     * 根据质控类型返回相应的字母
     *
     * @param
     * @return
     */
    public static String getQcType(String qcTypeNumber) {
        if (StringUtils.isBlank(qcTypeNumber) || Objects.equals("undefined", qcTypeNumber) || Objects.equals("null", qcTypeNumber) || Objects.equals("NULL", qcTypeNumber)) {
            return "";
        }
        String str = null;
        switch (qcTypeNumber) {
            case "0":
                str = "-PX";
                break;
            case "1":
                str = "-KB";
                break;
            default:
                str = "-ZK";
                break;
        }
        return str;
    }
    public static String getQcTypeFlag(String qcTypeStrForCode) {
        if (StringUtils.isBlank(qcTypeStrForCode) || Objects.equals("undefined", qcTypeStrForCode)
                || Objects.equals("null", qcTypeStrForCode) || Objects.equals("NULL", qcTypeStrForCode)) {
            return "";
        }
        return qcTypeStrForCode;
    }

    public static String getSubDate(String samplingDate) {
        if (StringUtils.isBlank(samplingDate) || Objects.equals("undefined", samplingDate)
                || Objects.equals("null", samplingDate) || Objects.equals("NULL", samplingDate) || samplingDate.length() != 10) {
            return "";
        }
        return samplingDate.replace("-", "").substring(2, 9);
    }

}
