/**
 * The MIT License (MIT)
 * <p>
 * Copyright (c) 2016 Caratacus
 * <p>
 * Permission is hereby granted, free of charge, to any person obtaining a copy of
 * this software and associated documentation files (the "Software"), to deal in
 * the Software without restriction, including without limitation the rights to
 * use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of
 * the Software, and to permit persons to whom the Software is furnished to do so,
 * subject to the following conditions:
 * <p>
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 * <p>
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS
 * FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR
 * COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
 * IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

package com.patzn.lims.common;

import com.patzn.lims.core.api.PtAssert;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.beans.Introspector;
import java.beans.PropertyDescriptor;
import java.lang.reflect.Array;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.sql.Timestamp;
import java.util.*;

/**
 * <p>
 * 对象工具类
 * <p>
 *
 * @author hubin
 * @since 2017-06-18
 */
public class ObjectUtils {

    private static Logger logger = LoggerFactory.getLogger(ObjectUtils.class);

    /**
     * 判断object是否为空,集合会校验size
     */
    public static boolean isNull(Object... objs) {
        for (Object obj : objs) {
            if (ObjectUtils.isEmpty(obj)) {
                return true;
            }
        }
        return false;
    }


    /**
     * 判断object是否不为空,集合会校验size
     */
    public static boolean isNotNull(Object... obj) {
        return !ObjectUtils.isNull(obj);
    }


    public static void checkNull(Object obj) {
        PtAssert.fail(null==obj,"数据有误");
    }

    public static void checkNull(Long [] ids) {
        PtAssert.fail(ArrayUtils.isEmpty(ids),"数据不能为空");
    }

    public static void checkNull(Long [] ids,String msg) {
        PtAssert.fail(ArrayUtils.isEmpty(ids),msg);
    }

    public static void checkNull(Object obj,String msg) {
        PtAssert.fail(null==obj,msg);
    }

    public static void checkNull(String str,String msg) {
        PtAssert.fail(StringUtils.isBlank(str),msg);
    }
    /**
     * 对象非空判断
     */
    public static boolean isEmpty(Object obj) {
        if (obj == null) {
            return true;
        }

        if (obj.getClass().isArray()) {
            return Array.getLength(obj) == 0;
        }
        if (obj instanceof CharSequence) {
            return ((CharSequence) obj).length() == 0;
        }
        if (obj instanceof Collection) {
            return ((Collection) obj).isEmpty();
        }
        if (obj instanceof Map) {
            return ((Map) obj).isEmpty();
        }
        // else
        return false;
    }


    /**
     * 比较两个实体属性值，返回一个map以有差异的属性名为key，value为一个list分别存obj1,obj2此属性名的值
     *
     * @param obj1       进行属性比较的对象1
     * @param obj2       进行属性比较的对象2
     * @param ignoreList 选择需要比较的属性数组
     * @return 属性差异比较结果map
     */
    public static Map<String, List<Object>> compareFields(Object obj1, Object obj2, List<String> ignoreList) {
        try {
            Map<String, List<Object>> map = new HashMap<>();
            if (CollectionUtils.isEmpty(ignoreList)) {
                return map;
            }
            if (obj1.getClass() == obj2.getClass()) {// 只有两个对象都是同一类型的才有可比性
                Class clazz = obj1.getClass();
                // 获取object的属性描述
                PropertyDescriptor[] pds = Introspector.getBeanInfo(clazz,
                        Object.class).getPropertyDescriptors();
                for (PropertyDescriptor pd : pds) {// 这里就是所有的属性了
                    String name = pd.getName();// 属性名
                    if (ignoreList != null && !ignoreList.contains(name)) {// 如果当前属性选择忽略比较，跳到下一次循环
                        continue;
                    }
                    Method readMethod = pd.getReadMethod();// get方法
                    // 在obj1上调用get方法等同于获得obj1的属性值
                    Object o1 = readMethod.invoke(obj1);
                    // 在obj2上调用get方法等同于获得obj2的属性值
                    Object o2 = readMethod.invoke(obj2);
                    if (o1 instanceof String && null != o1) {
                        o1 = ((String) o1).replaceAll("\n", "");
                    }
                    if (o2 instanceof String && null != o2) {
                        o2 = ((String) o2).replaceAll("\n", "");
                    }
                    if (o1 instanceof Timestamp) {
                        o1 = new Date(((Timestamp) o1).getTime());
                    }
                    if (o2 instanceof Timestamp) {
                        o2 = new Date(((Timestamp) o2).getTime());
                    }

                    if (o1 instanceof Date) {
                        o1 = DateUtils.toYearMonthDay((Date) o1);
                    }
                    if (o2 instanceof Date) {
                        o2 = DateUtils.toYearMonthDay((Date) o2);
                    }

                    if (o1 == null || "".equals(o1)) {
                        continue;
                    } else if (!Objects.equals(o1, o2)) {
                        List<Object> list = new ArrayList<>();
                        list.add(o1);
                        list.add(o2);
                        map.put(name, list);
                    }
                    /*else if (o1 == null && !"".equals(o2)) {
                        List<Object> list = new ArrayList<Object>();
                        list.add(o1);
                        list.add(o2);
                        map.put(name, list);
                        continue;
                    }
                    // 比较这两个值是否相等,不等就可以放入map了
                    if (!o1.equals(o2))*//* {
                        List<Object> list = new ArrayList<>();
                        list.add(o1);
                        list.add(o2);
                        map.put(name, list);
                    }*/
                }
                //这里是为了按照ignoreList的排序来拼接
                Map<String, List<Object>> returnMap = new LinkedHashMap<>();
                ignoreList.forEach(i -> {
                    List<Object> list = map.get(i);
                    if (CollectionUtils.isNotEmpty(list)) {
                        returnMap.put(i, list);
                    }
                });
                return returnMap;
            }
            return map;
        } catch (Exception e) {
            e.printStackTrace();
            logger.info(e.getMessage());
            return null;
        }
    }

    public static Map<String, List<Object>> compareTimeFormatFields(Object obj1, Object obj2, List<String> ignoreList) {
        try {
            Map<String, List<Object>> map = new HashMap<>();
            if (CollectionUtils.isEmpty(ignoreList)) {
                return map;
            }
            if (obj1.getClass() == obj2.getClass()) {// 只有两个对象都是同一类型的才有可比性
                Class clazz = obj1.getClass();
                // 获取object的属性描述
                PropertyDescriptor[] pds = Introspector.getBeanInfo(clazz,
                        Object.class).getPropertyDescriptors();
                for (PropertyDescriptor pd : pds) {// 这里就是所有的属性了
                    String name = pd.getName();// 属性名
                    if (ignoreList != null && !ignoreList.contains(name)) {// 如果当前属性选择忽略比较，跳到下一次循环
                        continue;
                    }
                    Method readMethod = pd.getReadMethod();// get方法
                    // 在obj1上调用get方法等同于获得obj1的属性值
                    Object o1 = readMethod.invoke(obj1);
                    // 在obj2上调用get方法等同于获得obj2的属性值
                    Object o2 = readMethod.invoke(obj2);
                    if (o1 instanceof Timestamp) {
                        o1 = new Date(((Timestamp) o1).getTime());
                    }
                    if (o2 instanceof Timestamp) {
                        o2 = new Date(((Timestamp) o2).getTime());
                    }

                    if (o1 instanceof Date) {
                        o1 = DateUtils.toYearMonthDayTimeFormat((Date) o1);
                    }
                    if (o2 instanceof Date) {
                        o2 = DateUtils.toYearMonthDayTimeFormat((Date) o2);
                    }

                    if ((o1 == null || "".equals(o1)) && (o2 == null || "".equals(o2))) {
                        continue;
                    } else if (!Objects.equals(o1, o2)) {
                        List<Object> list = new ArrayList<>();
                        list.add(o1);
                        list.add(o2);
                        map.put(name, list);
                    }
                    /*else if (o1 == null && !"".equals(o2)) {
                        List<Object> list = new ArrayList<Object>();
                        list.add(o1);
                        list.add(o2);
                        map.put(name, list);
                        continue;
                    }
                    // 比较这两个值是否相等,不等就可以放入map了
                    if (!o1.equals(o2))*//* {
                        List<Object> list = new ArrayList<>();
                        list.add(o1);
                        list.add(o2);
                        map.put(name, list);
                    }*/
                }
            }
            return map;
        } catch (Exception e) {
            e.printStackTrace();
            logger.info(e.getMessage());
            return null;
        }
    }

    /**
     * 比较两个实体属性值，返回一个map以有差异的属性名为key，value为一个list分别存obj1,obj2此属性名的值
     *
     * @param newOne     进行属性比较的对象1
     * @param oldOne     进行属性比较的对象2
     * @param ignoreList 选择需要比较的属性数组
     * @return 属性差异比较结果map
     */
    public static Map<String, List<Object>> compareFieldsDiffent(Object newOne, Object oldOne, List<String> ignoreList) {
        try {
            Map<String, List<Object>> map = new HashMap<>();
            if (CollectionUtils.isEmpty(ignoreList)) {
                return map;
            }
            if (newOne.getClass() == oldOne.getClass()) {// 只有两个对象都是同一类型的才有可比性
                Class clazz = newOne.getClass();
                // 获取object的属性描述
                PropertyDescriptor[] pds = Introspector.getBeanInfo(clazz,
                        Object.class).getPropertyDescriptors();
                for (PropertyDescriptor pd : pds) {// 这里就是所有的属性了
                    String name = pd.getName();// 属性名
                    if (ignoreList != null && !ignoreList.contains(name)) {// 如果当前属性选择忽略比较，跳到下一次循环
                        continue;
                    }
                    Method readMethod = pd.getReadMethod();// get方法
                    // 在obj1上调用get方法等同于获得obj1的属性值
                    Object o1 = readMethod.invoke(newOne);

                    if (null == o1) {
                        continue;
                        //如果界面上的值是空就不比较了
                    }
                    // 在obj2上调用get方法等同于获得obj2的属性值
                    Object o2 = readMethod.invoke(oldOne);
                    if (o1 instanceof Timestamp) {
                        o1 = new Date(((Timestamp) o1).getTime());
                    }
                    if (o2 instanceof Timestamp) {
                        o2 = new Date(((Timestamp) o2).getTime());
                    }

                    if (o1 instanceof Date) {
                        o1 = DateUtils.toYearMonthDay((Date) o1);
                    }
                    if (o2 instanceof Date) {
                        o2 = DateUtils.toYearMonthDay((Date) o2);
                    }

                    if ((o1 == null || "".equals(o1)) && (o2 == null || "".equals(o2))) {
                        continue;
                    } else if (!Objects.equals(o1, o2)) {
                        List<Object> list = new ArrayList<>();
                        list.add(o1);
                        list.add(o2);
                        map.put(name, list);
                    }

                }
            }
            return map;
        } catch (Exception e) {
            e.printStackTrace();
            return null;
        }
    }

    public static void setProperty(Object base, String baseProperty, Object source, String sourceProperty) {
        if (base == null || StringUtils.isEmpty(baseProperty) || source == null || StringUtils.isEmpty(sourceProperty)) {
            return;
        }
        // base对象的baseProperty的set方法
        String baseSet = "set" + baseProperty.substring(0, 1).toUpperCase() + baseProperty.substring(1);
        Method baseSetter = null;
        try {
            baseSetter = base.getClass().getMethod(baseSet);
        } catch (NoSuchMethodException ignored) {
        }
        if (baseSetter == null) {
            return;
        }
        // source对象的sourceProperty的get方法
        String sourceGet = "get" + sourceProperty.substring(0, 1).toUpperCase() + sourceProperty.substring(1);
        Method sourceGetter = null;
        try {
            sourceGetter = source.getClass().getMethod(sourceGet);
        } catch (NoSuchMethodException ignored) {
        }
        if (sourceGetter == null) {
            return;
        }
        try {
            baseSetter.invoke(base, sourceGetter.invoke(source));
        } catch (IllegalAccessException | InvocationTargetException ignored) {
        }
    }

}
